<?php
declare(strict_types=1);

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/functions/utils.php';
require_once __DIR__ . '/functions/telegram.php';
require_once __DIR__ . '/functions/storage.php';
require_once __DIR__ . '/functions/users.php';
require_once __DIR__ . '/functions/acc.php';
require_once __DIR__ . '/handlers/start.php';
require_once __DIR__ . '/handlers/ceo.php';
require_once __DIR__ . '/handlers/accounting.php';
require_once __DIR__ . '/handlers/accounting_bind.php';
require_once __DIR__ . '/handlers/hook_accounting_text.php';

function vm_log(string $line): void {
    $file = LOG_DIR . '/' . date('Y-m-d') . '.log';
    $line = '[' . date('H:i:s') . '] ' . $line . PHP_EOL;
    error_log($line, 3, $file);
}

// ---------- healthcheck ----------
if (isset($_GET['ping'])) {
    header('Content-Type: application/json; charset=UTF-8');
    echo json_encode([
        'ok' => true,
        'ts' => date('Y-m-d H:i:s'),
        'php' => PHP_VERSION,
        'bundle' => 'FINAL_v1',
        'token_len' => strlen(BOT_TOKEN),
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

// ---------- read update ----------
$raw = file_get_contents('php://input');
if ($raw !== '' && $raw !== false) {
    vm_log('RAW ' . $raw);
}
$update = $raw ? json_decode($raw, true) : null;

function is_ceo(int $uid): bool { return $uid === CEO_ID; }

function handle_message(array $msg): void {
    $uid = (int)($msg['from']['id'] ?? 0);
    $chat_id = (int)($msg['chat']['id'] ?? 0);
    $text = trim((string)($msg['text'] ?? ''));

    // اتصال متن‌های منوی فارسی به ماژول حسابداری (در صورت match توقف)
    if (function_exists('acc_bind_handle_text') && acc_bind_handle_text($uid, $chat_id, $text)) { return; }

    if ($text === '/start' || $text === 'start') {
        start_menu($uid);
        return;
    }
    if ($text === '/ceo' && is_ceo($uid)) {
        ceo_menu($uid);
        return;
    }
    if ($text === '/acc' && is_ceo($uid)) {
        if (function_exists('acc_menu')) { acc_menu($uid, $chat_id, 0); } else { acc_menu($uid); }
        return;
    }
    tg_send_html($chat_id, '❗️دستور نامعتبر است. از /start استفاده کنید.');
}

function handle_callback(array $cb): void {
    $uid = (int)($cb['from']['id'] ?? 0);
    $data = (string)($cb['data'] ?? '');
    $chat_id = (int)($cb['message']['chat']['id'] ?? 0);
    $mid = (int)($cb['message']['message_id'] ?? 0);

    if (strpos($data, 'start:') === 0) {
        start_handle($uid, $data, $chat_id, $mid);
        return;
    }
    if (strpos($data, 'ceo:') === 0 && is_ceo($uid)) {
        ceo_handle($uid, $data, $chat_id, $mid);
        return;
    }
    if (strpos($data, 'acc:') === 0 && is_ceo($uid)) {
        acc_handle($uid, $data, $chat_id, $mid);
        return;
    }

    if (isset($cb['id'])) tg_answer_cb((string)$cb['id'], 'اجازه دسترسی ندارید.', true);
}

if (is_array($update)) {
    // قلاب زودهنگام: قبل از هندلرهای قدیمی، متن‌های منوی فارسی را به حسابداری هدایت می‌کند
    if (function_exists('hook_accounting_text') && hook_accounting_text($update)) { return; }

    if (isset($update['message'])) {
        handle_message($update['message']);
    } elseif (isset($update['callback_query'])) {
        handle_callback($update['callback_query']);
    } else {
        vm_log('Unknown update structure');
    }
} else {
    header('Content-Type: text/html; charset=UTF-8');
    echo '<!doctype html><meta charset="utf-8"><title>Vision Media Bot</title>';
    echo '<h3>Vision Media Bot — FINAL v1</h3>';
    echo '<p>Ping: <code>?ping</code></p>';
}
