<?php
declare(strict_types=1);

/** Vision Media Bot — index.php (FULL, stable, impersonation-aware) */

error_reporting(E_ALL);
ini_set('display_errors', '0');

require_once __DIR__ . '/config.php';
if (!is_dir(LOG_DIR)) @mkdir(LOG_DIR, 0775, true);
ini_set('log_errors', '1');
ini_set('error_log', LOG_DIR . '/php-error.log');

/* ===== Core libs ===== */
require_once __DIR__ . '/functions/utils.php';
require_once __DIR__ . '/functions/telegram.php';
require_once __DIR__ . '/functions/conversation.php';
require_once __DIR__ . '/functions/storage.php';
require_once __DIR__ . '/functions/users.php';
require_once __DIR__ . '/functions/approvals.php';
require_once __DIR__ . '/functions/penalties.php';
require_once __DIR__ . '/functions/tasks.php';
require_once __DIR__ . '/functions/calendar.php';
require_once __DIR__ . '/functions/accounting.php';
require_once __DIR__ . '/router.php';

/* ===== Handlers ===== */
require_once __DIR__ . '/handlers/start.php';
require_once __DIR__ . '/handlers/ceo.php';
require_once __DIR__ . '/handlers/accounting.php';   // acc_conv_dispatch_text
require_once __DIR__ . '/handlers/department.php';   // dept_conv_dispatch_text + presence/SLA cron
require_once __DIR__ . '/handlers/admin.php';
require_once __DIR__ . '/handlers/employee.php';     // employee_conv_dispatch_text + editor buttons
require_once __DIR__ . '/handlers/customer.php';     // customer_conv_dispatch_text
require_once __DIR__ . '/handlers/penalties.php';

/* ===== Health & Cron ===== */
if (isset($_GET['ping'])) { echo "OK"; exit; }

if (isset($_GET['cron'])) {
    acc_reminders_run();     // پرداخت‌ها (سررسید/دیرکرد)
    task_send_reminders();   // یادآور وظایف (5 ساعت مانده/1 ساعت مانده)
    task_check_overdue();    // جریمه‌ی تأخیر وظایف
    task_cleanup();          // پاکسازی پیام‌های یادآور قدیمی
    dept_presence_cron();    // پرسش حضور + فعال‌سازی دکمه اتمام پس از 30 دقیقه
    dept_sla_cron();         // SLA ۷روزه‌ی اولین آفیش
    echo "CRON-OK";
    exit;
}

/* ===== Read Telegram update ===== */
$raw = file_get_contents('php://input');
if (!$raw) { echo "NO-UPDATE"; exit; }

file_put_contents(
    LOG_DIR . '/router-' . date('Y-m-d') . '.log',
    "[" . date('H:i:s') . "] " . $raw . "\n",
    FILE_APPEND
);

$update  = json_decode($raw, true);
$chat_id = null; $text = null; $data = null; $from_id = null; $name = null;

if (isset($update['message'])) {
    $m = $update['message'];
    $chat_id = (int)($m['chat']['id'] ?? 0);
    $from_id = (int)($m['from']['id'] ?? 0);
    $name    = trim(($m['from']['first_name'] ?? '') . ' ' . ($m['from']['last_name'] ?? ''));
    $text    = $m['text'] ?? '';
}
if (isset($update['callback_query'])) {
    $cq = $update['callback_query'];
    $data    = $cq['data'] ?? '';
    $from_id = (int)($cq['from']['id'] ?? 0);
    $name    = trim(($cq['from']['first_name'] ?? '') . ' ' . ($cq['from']['last_name'] ?? ''));
    $chat_id = (int)($cq['message']['chat']['id'] ?? 0);
}

if (!$from_id || !$chat_id) { echo "BAD"; exit; }

/* ===== Ensure user & CEO ===== */
$user = user_get($from_id);
if (!$user) {
    $user = ['id'=>$from_id, 'full_name'=>$name, 'role'=>null, 'created'=>time()];
    user_save($user);
}
if ($from_id === CEO_ID && ($user['role'] ?? '') !== 'CEO') {
    $user['role'] = 'CEO';
    user_save($user);
}

/* ===== Helper: effective user during CEO impersonation ===== */
function effective_user_for_ceo(int $from_id, array $user): array {
    if (is_ceo($user) && conv_in($from_id, 'impersonate')) {
        $st = conv_get($from_id);
        $target = (int)($st['data']['cust_id'] ?? 0);
        if ($target > 0) return user_get($target) ?: ['id'=>$target,'full_name'=>'کاربر','role'=>null];
    }
    return $user;
}

/* ===== Dispatch ===== */
if ($text) {
    // Conversation flows (order matters)
    if (function_exists('acc_conv_dispatch_text')      && acc_conv_dispatch_text($chat_id, $from_id, $text)) { echo "OK"; exit; }
    if (function_exists('dept_conv_dispatch_text')     && dept_conv_dispatch_text($chat_id, $from_id, $text)) { echo "OK"; exit; }
    if (function_exists('employee_conv_dispatch_text') && employee_conv_dispatch_text($chat_id, $from_id, $text)) { echo "OK"; exit; }
    if (function_exists('customer_conv_dispatch_text') && customer_conv_dispatch_text($chat_id, $from_id, $text)) { echo "OK"; exit; }

    if (preg_match('/^\\/start/i', $text)) {
        handle_start($chat_id, $from_id, $user);
    } elseif (preg_match('/^\\/ping/i', $text)) {
        tg_send_html($chat_id, "pong 🏓");
    } elseif (!conv_dispatch($chat_id, $from_id, $text)) {
        tg_send_html($chat_id, vm_polite_header($user['full_name']) . "از منوی زیر استفاده کن عزیزم 👇");
        handle_start($chat_id, $from_id, $user);
    }
}
elseif ($data) {
    // دکمه‌های ویژه تدوینگر (OK/NOK) را قبل از سوئیچ روت، به‌صورت عمومی هندل می‌کنیم
    if (function_exists('employee_router_check_buttons') && employee_router_check_buttons($data, $chat_id, $from_id)) {
        http_response_code(200); echo "OK"; exit;
    }

    $parts = explode(':', $data);
    $root  = $parts[0] ?? '';

    switch ($root) {
        case 'acc': {
            $eff = effective_user_for_ceo($from_id, $user);
            acc_router($chat_id, (int)$eff['id'], $eff, $parts);
            break;
        }
        case 'dept': {
            $eff = effective_user_for_ceo($from_id, $user);
            dept_router($chat_id, (int)$eff['id'], $eff, $parts);
            break;
        }
        case 'admin': {
            $eff = effective_user_for_ceo($from_id, $user);
            admin_router($chat_id, (int)$eff['id'], $eff, $parts);
            break;
        }
        case 'emp': {
            $eff = effective_user_for_ceo($from_id, $user);
            employee_router($chat_id, (int)$eff['id'], $eff, $parts);
            break;
        }
        case 'cust': {
            $eff = effective_user_for_ceo($from_id, $user);
            customer_router($chat_id, (int)$eff['id'], $eff, $parts);
            break;
        }
        case 'ceo':
            ceo_router($chat_id, $from_id, $user, $parts);
            break;

        case 'pen':
            penalties_router($chat_id, $from_id, $user, $parts);
            break;

        default:
            tg_send_html($chat_id, "دستور نامعتبر ❌");
    }
}

http_response_code(200);
echo "OK";
