<?php
declare(strict_types=1);

/**
 * Vision Media Bot — storage.php (Part 1/7)
 * توابع ذخیره‌سازی عمومی برای JSON
 */

// --- Core JSON IO ---
function storage_load(string $file): array {
    if (!file_exists($file)) return [];
    $txt = file_get_contents($file);
    $d   = json_decode($txt, true);
    return $d ?: [];
}

function storage_save(string $file, array $data): void {
    file_put_contents($file, json_encode($data, JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT));
}
// --- Append helper ---
function storage_append(string $file, array $row): void {
    $all = storage_load($file);
    $all[] = $row;
    storage_save($file, $all);
}

// --- Find helper ---
function storage_find(string $file, callable $fn): ?array {
    $all = storage_load($file);
    foreach ($all as $row) {
        if ($fn($row)) return $row;
    }
    return null;
}
// --- Update helper ---
function storage_update(string $file, callable $fn): void {
    $all = storage_load($file);
    $changed = false;
    foreach ($all as &$row) {
        $new = $fn($row);
        if ($new !== null) {
            $row = $new;
            $changed = true;
        }
    }
    if ($changed) storage_save($file, $all);
}

// --- Delete helper ---
function storage_delete(string $file, callable $fn): void {
    $all = storage_load($file);
    $new = [];
    foreach ($all as $row) {
        if (!$fn($row)) $new[] = $row;
    }
    storage_save($file, $new);
}
// --- ID-based helpers ---
function storage_get_by_id(string $file, string $id): ?array {
    $all = storage_load($file);
    foreach ($all as $row) {
        if (($row['id'] ?? '') === $id) return $row;
    }
    return null;
}

function storage_update_by_id(string $file, string $id, array $newData): void {
    $all = storage_load($file);
    foreach ($all as &$row) {
        if (($row['id'] ?? '') === $id) {
            $row = array_merge($row, $newData);
        }
    }
    storage_save($file, $all);
}

function storage_delete_by_id(string $file, string $id): void {
    $all = storage_load($file);
    $new = array_values(array_filter($all, fn($r) => ($r['id'] ?? '') !== $id));
    storage_save($file, $new);
}
// --- Ensure file exists ---
function storage_ensure(string $file): void {
    $dir = dirname($file);
    if (!is_dir($dir)) @mkdir($dir, 0775, true);
    if (!file_exists($file)) {
        file_put_contents($file, '[]');
    }
}

// --- Reset storage (dangerous) ---
function storage_reset(string $file): void {
    file_put_contents($file, '[]');
}
// --- Search helpers ---
function storage_search(string $file, string $q, array $fields): array {
    $all = storage_load($file);
    $out = [];
    foreach ($all as $row) {
        foreach ($fields as $f) {
            if (isset($row[$f]) && str_has((string)$row[$f], $q)) {
                $out[] = $row;
                break;
            }
        }
    }
    return $out;
}

function storage_filter(string $file, callable $fn): array {
    $all = storage_load($file);
    return array_values(array_filter($all, $fn));
}
// --- Generic incremental ID storage ---
function storage_next_id(string $file, string $key='id'): int {
    $all = storage_load($file);
    $max = 0;
    foreach ($all as $row) {
        $val = (int)($row[$key] ?? 0);
        if ($val > $max) $max = $val;
    }
    return $max + 1;
}

// --- پایان storage.php ---
