<?php
declare(strict_types=1);
require_once __DIR__ . '/storage.php';
require_once __DIR__ . '/telegram.php';
require_once __DIR__ . '/utils.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/tasks.php';

/*
 * Teams: per customer_id => manager_id, members: scenario, blogger, photographer, editor, admin
 * Offish: id, customer_id, manager_id, date_ts, location, address, ig_handle, notify_sent, confirms (photographer/blogger), arrived checks, finished_ts
 * Quota: 3 per Jalali month (simplified here as per Gregorian month for demo); extra requires accounting approval (to be integrated via approvals)
 */

function team_set(int $customer_id, int $manager_id, array $members): void {
    $t = teams_all(); $t[(string)$customer_id] = ['customer_id'=>$customer_id,'manager_id'=>$manager_id,'members'=>$members,'created_at'=>time()];
    teams_save($t);
}
function team_get(int $customer_id): array {
    $t = teams_all(); return $t[(string)$customer_id] ?? [];
}

function offish_create(int $customer_id, int $manager_id, int $date_ts, array $info): int {
    $o = offish_all();
    $id = (int)($o['last_id'] ?? 0) + 1;
    $o['last_id'] = $id;
    $o['items'][] = [
        'id'=>$id,'customer_id'=>$customer_id,'manager_id'=>$manager_id,'date_ts'=>$date_ts,
        'info'=>$info,'status'=>'scheduled','created_at'=>time(),
        'confirms'=>['photographer'=>null,'blogger'=>null], 'arrived'=>['photographer'=>null,'blogger'=>null],
        'finished_ts'=>0
    ];
    offish_save($o);

    // Notifications to photographer + blogger, with inline confirm/decline
    $team = team_get($customer_id);
    $ph = (int)($team['members']['photographer'] ?? 0);
    $bl = (int)($team['members']['blogger'] ?? 0);
    $msg = "📅 آفیش جدید ثبت شد:\n"
         ."مشتری: ".($info['customer_name']??'-')."\n"
         ."پیج: ".($info['ig']??'-')."\n"
         ."آدرس: ".($info['address']??'-')."\n"
         ."لوکیشن: ".($info['location']??'-')."\n"
         ."تاریخ/ساعت: ".date('Y-m-d H:i', $date_ts)."\n\n"
         ."لطفاً حضور را تأیید یا رد کنید.";
    $kb = [[ kb_btn('تأیید ✅','offish:confirm:yes:'.$id), kb_btn('رد ❌','offish:confirm:no:'.$id) ]];
    if ($ph) tg_send_html($ph, $msg, $kb);
    if ($bl) tg_send_html($bl, $msg, $kb);
    return $id;
}

function offish_get(int $id): array {
    $o = offish_all(); foreach (($o['items']??[]) as $it) if ((int)$it['id']===$id) return $it; return [];
}
function offish_update(array $rec): void {
    $o = offish_all(); foreach (($o['items']??[]) as &$it) if ((int)$it['id']===(int)$rec['id']) { $it=$rec; } offish_save($o);
}

function month_key(int $ts): string { return date('Y-m', $ts); }
function offish_count_month(int $customer_id, string $mkey): int {
    $o = offish_all(); $n=0;
    foreach (($o['items']??[]) as $it) if ((int)$it['customer_id']===$customer_id && month_key((int)$it['date_ts'])===$mkey) $n++;
    return $n;
}
function offish_can_schedule(int $customer_id, int $date_ts): bool {
    return offish_count_month($customer_id, month_key($date_ts)) < 3;
}
function offish_arrival_prompt(int $offish_id): void {
    $rec = offish_get($offish_id); if (!$rec) return;
    $team = team_get((int)$rec['customer_id']);
    foreach (['photographer','blogger'] as $role) {
        $uid = (int)($team['members'][$role] ?? 0); if (!$uid) continue;
        $kb = [[ kb_btn("حاضر هستم ✅","offish:arrive:yes:".$offish_id.":".$role), kb_btn("نرسیدم ❌","offish:arrive:no:".$offish_id.":".$role) ]];
        tg_send_html($uid, "📍 آیا در آفیش #".$offish_id." حاضر هستی؟", $kb);
    }
}
function offish_mark_arrival(int $offish_id, string $role, bool $yes): void {
    $rec = offish_get($offish_id); if (!$rec) return;
    $rec['arrived'][$role] = $yes ? time() : 0;
    if ($yes) $rec['start_ts'] = time();
    offish_update($rec);
}
function offish_finish_allowed(int $offish_id): bool {
    $rec = offish_get($offish_id); if (!$rec) return false;
    $st = (int)($rec['start_ts'] ?? 0); if (!$st) return false;
    return time() >= ($st + 30*60);
}
