<?php
declare(strict_types=1);

/**
 * /bot/core/jalali.php
 * توابع تاریخ جلالی (بدون وابستگی خارجی)
 * - تبدیل میلادی⇄جلالی
 * - jdate: مشابه date() ولی با خروجی شمسی
 */

// نقشه اعداد فارسی
const J_NUM_FA = ['0'=>'۰','1'=>'۱','2'=>'۲','3'=>'۳','4'=>'۴','5'=>'۵','6'=>'۶','7'=>'۷','8'=>'۸','9'=>'۹'];

// نام ماه‌ها و روزها
const J_MONTHS_FA = ['فروردین','اردیبهشت','خرداد','تیر','مرداد','شهریور','مهر','آبان','آذر','دی','بهمن','اسفند'];
const J_WEEKDAYS_FA = ['یکشنبه','دوشنبه','سه‌شنبه','چهارشنبه','پنجشنبه','جمعه','شنبه'];
// ───────────────────────────
// تبدیل میلادی به جلالی
// ───────────────────────────
if (!function_exists('gregorian_to_jalali')) {
    function gregorian_to_jalali(int $gy, int $gm, int $gd): array {
        $g_d_m = [0,31,59,90,120,151,181,212,243,273,304,334];
        $gy2 = $gy - 1600;
        $gm2 = $gm - 1;
        $gd2 = $gd - 1;

        $g_day_no = 365*$gy2 + intval(($gy2+3)/4) - intval(($gy2+99)/100) + intval(($gy2+399)/400);
        $g_day_no += $g_d_m[$gm2] + $gd2;

        // leap
        if ($gm2>1 && (($gy%4==0 && $gy%100!=0) || ($gy%400==0))) $g_day_no++;

        $j_day_no = $g_day_no - 79;

        $j_np = intval($j_day_no / 12053); // 12053 = 365*33 + 8
        $j_day_no %= 12053;

        $jy = 979 + 33*$j_np + 4*intval($j_day_no/1461);
        $j_day_no %= 1461;

        if ($j_day_no >= 366) {
            $jy += intval(($j_day_no-366)/365);
            $j_day_no = ($j_day_no-366)%365;
        }

        $jm = 0;
        for ($i=0; $i<11 && $j_day_no >= ( ($i<6)?31:30 ); $i++) {
            $j_day_no -= ($i<6)?31:30;
            $jm++;
        }
        $jm++;
        $jd = $j_day_no + 1;

        return [$jy, $jm, $jd];
    }
}
// ───────────────────────────
// تبدیل جلالی به میلادی
// ───────────────────────────
if (!function_exists('jalali_to_gregorian')) {
    function jalali_to_gregorian(int $jy, int $jm, int $jd): array {
        $jy -= 979;
        $jm -= 1;
        $jd -= 1;

        $j_day_no = 365*$jy + intval($jy/33)*8 + intval(($jy%33 +3)/4);

        for ($i=0; $i<$jm; ++$i) {
            $j_day_no += ($i<6)?31:30;
        }
        $j_day_no += $jd;

        $g_day_no = $j_day_no + 79;

        $gy = 1600 + 400*intval($g_day_no/146097); // 146097 = 365*400 + 400/4 - 400/100 + 400/400
        $g_day_no %= 146097;

        $leap = true;
        if ($g_day_no >= 36525) { // 36525 = 365*100 + 100/4
            $g_day_no--;
            $gy += 100*intval($g_day_no/36524);
            $g_day_no %= 36524;

            if ($g_day_no >= 365) $g_day_no++;
            else $leap = false;
        }

        $gy += 4*intval($g_day_no/1461);
        $g_day_no %= 1461;

        if ($g_day_no >= 366) {
            $leap = false;
            $g_day_no--;
            $gy += intval($g_day_no/365);
            $g_day_no %= 365;
        }

        $gm_days = [0,31, ($leap?29:28),31,30,31,30,31,31,30,31,30,31];
        $gm = 0;
        for ($i=1; $i<=12 && $g_day_no >= $gm_days[$i]; $i++) {
            $g_day_no -= $gm_days[$i];
            $gm = $i;
        }
        $gm++;
        $gd = $g_day_no + 1;

        return [$gy, $gm, $gd];
    }
}
// ───────────────────────────
// کمک‌یار نمایش فارسی
// ───────────────────────────
if (!function_exists('fa_num')) {
    function fa_num(string $s): string {
        return strtr($s, J_NUM_FA);
    }
}

if (!function_exists('fa_pad2')) {
    function fa_pad2(int $n): string {
        return fa_num(str_pad((string)$n, 2, '0', STR_PAD_LEFT));
    }
}
// ───────────────────────────
// jdate(format, timestamp)
// ───────────────────────────
if (!function_exists('jdate')) {
    /**
     * jdate — فرمت مشابه date() ولی بر اساس جلالی
     * پترن‌های رایج: Y(سال چهاررقمی)،y،m،n،d،j،H،i،s،D(نام کوتاه)،l(نام کامل)،F(نام ماه)،M(کوتاه ماه)
     */
    function jdate(string $format, ?int $ts = null): string {
        if ($ts === null) $ts = time();

        $gy = (int)date('Y', $ts);
        $gm = (int)date('n', $ts);
        $gd = (int)date('j', $ts);
        [$jy, $jm, $jd] = gregorian_to_jalali($gy, $gm, $gd);

        $w = (int)date('w', $ts); // 0:یکشنبه تا 6:شنبه (با توجه به PHP)
        // در PHP: 0=Sunday; ما آرایهٔ J_WEEKDAYS_FA را به ترتیب [یکشنبه..شنبه] نگه داشتیم
        $weekdayFull = J_WEEKDAYS_FA[$w];
        $weekdayShort = mb_substr($weekdayFull, 0, 3);

        $monthFull = J_MONTHS_FA[$jm-1];
        $monthShort = mb_substr($monthFull, 0, 3);

        $out = '';
        $len = mb_strlen($format);
        for ($i=0; $i<$len; $i++) {
            $ch = mb_substr($format, $i, 1);
            switch ($ch) {
                case '\\': // escape next char
                    $i++;
                    if ($i < $len) $out .= mb_substr($format, $i, 1);
                    break;
                case 'Y': $out .= fa_num((string)$jy); break;
                case 'y': $out .= fa_num(substr((string)$jy, -2)); break;
                case 'm': $out .= fa_pad2($jm); break;
                case 'n': $out .= fa_num((string)$jm); break;
                case 'd': $out .= fa_pad2($jd); break;
                case 'j': $out .= fa_num((string)$jd); break;
                case 'H': $out .= fa_num(date('H', $ts)); break;
                case 'i': $out .= fa_num(date('i', $ts)); break;
                case 's': $out .= fa_num(date('s', $ts)); break;
                case 'D': $out .= $weekdayShort; break;
                case 'l': $out .= $weekdayFull; break;
                case 'F': $out .= $monthFull; break;
                case 'M': $out .= $monthShort; break;
                default:  $out .= $ch; break;
            }
        }
        return $out;
    }
}
// ───────────────────────────
// jstrtotime: ساخت timestamp از تاریخ شمسی
// ورودی مثل "1404-06-15 14:30:00" یا "1404/6/15"
// ───────────────────────────
if (!function_exists('jstrtotime')) {
    function jstrtotime(string $jdate, string $time = '00:00:00'): ?int {
        $jdate = trim($jdate);
        $jdate = str_replace(['/', '.', '_'], '-', $jdate);
        $parts = explode(' ', $jdate);
        $datePart = $parts[0] ?? '';
        $timePart = $parts[1] ?? $time;

        [$jy,$jm,$jd] = array_map('intval', explode('-', $datePart));
        [$gy,$gm,$gd] = jalali_to_gregorian($jy, $jm, $jd);

        [$H,$i,$s] = array_map('intval', explode(':', $timePart));
        return mktime($H, $i, $s, $gm, $gd, $gy);
    }
}
// ───────────────────────────
// پایان
// ───────────────────────────
return true;
