<?php
declare(strict_types=1);

/**
 * Vision Media Bot — conversation.php (Part 1/7)
 * مدیریت وضعیت‌های موقت گفتگو
 */

const CONV_FILE = __DIR__ . '/../storage/conversations.json';

// --- Load / Save ---
function conv_load(): array {
    if (!file_exists(CONV_FILE)) return [];
    $txt = file_get_contents(CONV_FILE);
    $d   = json_decode($txt, true);
    return $d ?: [];
}
function conv_save(array $d): void {
    file_put_contents(CONV_FILE, json_encode($d, JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT));
}
// --- Get / Set user state ---
function conv_get(int $uid): ?array {
    $all = conv_load();
    return $all[$uid] ?? null;
}

function conv_set(int $uid, array $state): void {
    $all = conv_load();
    $all[$uid] = $state;
    conv_save($all);
}

function conv_clear(int $uid): void {
    $all = conv_load();
    unset($all[$uid]);
    conv_save($all);
}
// --- Helpers for step-based forms ---
function conv_start(int $uid, string $flow, int $step=0, array $data=[]): void {
    conv_set($uid, [
        'flow' => $flow,
        'step' => $step,
        'data' => $data,
        'time' => time(),
    ]);
}

function conv_next(int $uid, array $state): void {
    $state['step'] = ($state['step'] ?? 0) + 1;
    $state['time'] = time();
    conv_set($uid, $state);
}

function conv_back(int $uid, array $state): void {
    $state['step'] = max(0, ($state['step'] ?? 0) - 1);
    $state['time'] = time();
    conv_set($uid, $state);
}
// --- Flow check ---
function conv_in(int $uid, string $flow): bool {
    $s = conv_get($uid);
    return $s && ($s['flow'] ?? '') === $flow;
}

function conv_step(int $uid): int {
    $s = conv_get($uid);
    return (int)($s['step'] ?? 0);
}

function conv_data(int $uid): array {
    $s = conv_get($uid);
    return $s['data'] ?? [];
}

function conv_update_data(int $uid, array $newData): void {
    $s = conv_get($uid);
    if (!$s) return;
    $s['data'] = array_merge($s['data'] ?? [], $newData);
    $s['time'] = time();
    conv_set($uid, $s);
}
// --- Conversation Expiry ---
const CONV_TTL = 3600; // 1 ساعت

function conv_gc(): void {
    $all = conv_load();
    $changed = false;
    foreach ($all as $uid => $s) {
        if (time() - ($s['time'] ?? 0) > CONV_TTL) {
            unset($all[$uid]);
            $changed = true;
        }
    }
    if ($changed) conv_save($all);
}

// --- Conversation Prompt Helper ---
function conv_prompt(int $chat_id, int $uid, string $msg, string $flow, int $step, array $data=[]): void {
    conv_start($uid, $flow, $step, $data);
    tg_send_html($chat_id, $msg);
}
// --- Multi-step Example Handler ---
function conv_handle_example(int $chat_id, int $uid, string $text): void {
    $s = conv_get($uid);
    if (!$s || ($s['flow'] ?? '') !== 'example') return;

    $step = $s['step'] ?? 0;
    $data = $s['data'] ?? [];

    if ($step === 0) {
        $data['name'] = $text;
        conv_start($uid, 'example', 1, $data);
        tg_send_html($chat_id, "خب {$text} عزیز 🌹، سنت چند ساله؟");
    }
    elseif ($step === 1) {
        $data['age'] = $text;
        conv_clear($uid);
        tg_send_html($chat_id,
            "ممنون 🙏\n"
            . "اسم: {$data['name']}\n"
            . "سن: {$data['age']}"
        );
    }
}
// --- Dispatcher for conversations ---
function conv_dispatch(int $chat_id, int $uid, string $text): bool {
    $s = conv_get($uid);
    if (!$s) return false;

    // نمونه: example flow
    if (($s['flow'] ?? '') === 'example') {
        conv_handle_example($chat_id, $uid, $text);
        return true;
    }

    return false;
}

// --- پایان conversation.php ---
