<?php
declare(strict_types=1);
require_once __DIR__ . '/storage.php';
require_once __DIR__ . '/users.php';
require_once __DIR__ . '/tasks.php';
require_once __DIR__ . '/penalties.php';
require_once __DIR__ . '/offish.php';
require_once __DIR__ . '/utils.php';

function vm_period_range(string $period): array {
    // returns [start_ts, end_ts]
    $now = time();
    $end = $now;
    if ($period==='day') {
        $s = strtotime("today 00:00:00");
        return [$s, $end];
    } elseif ($period==='week') {
        // week starting Saturday (Iran common) – use last Saturday 00:00
        $dow = (int)date('w'); // 0=Sun
        $daysSinceSat = ( ($dow+1)%7 ); // Sat=6 -> 0
        $s = strtotime("-".$daysSinceSat." days 00:00:00");
        return [$s, $end];
    } elseif ($period==='month') {
        $s = strtotime(date('Y-m-01 00:00:00'));
        return [$s, $end];
    } elseif ($period==='prev_week') {
        $dow = (int)date('w');
        $daysSinceSat = ( ($dow+1)%7 );
        $thisSat = strtotime("-".$daysSinceSat." days 00:00:00");
        $prevSat = strtotime("-7 days", $thisSat);
        return [$prevSat, $thisSat];
    } elseif ($period==='prev_month') {
        $firstThis = strtotime(date('Y-m-01 00:00:00'));
        $firstPrev = strtotime("-1 month", $firstThis);
        return [$firstPrev, $firstThis];
    }
    return [0, $end];
}

function vm_report_employee_text(int $uid, string $period): string {
    [$start, $end] = vm_period_range($period);
    $user = users_get($uid);
    $name = $user['first_name'] ?? ('#'.$uid);
    // Load storages
    $tasks = tasks_all();
    $pens = penalties_all();
    $offishes = offish_all(); // must return list; if undefined wrap
    if (!is_array($offishes)) $offishes = [];
    // Aggregates
    $task_done = 0; $task_total = 0; $task_overdue = 0;
    foreach (($tasks['items'] ?? []) as $t) {
        if ((int)($t['assignee_id'] ?? 0) !== $uid) continue;
        $ts = (int)($t['created_at'] ?? 0);
        if ($ts < $start || $ts > $end) continue;
        $task_total++;
        if (!empty($t['done_at'])) $task_done++;
        if (!empty($t['due_at']) && empty($t['done_at']) && time() > (int)$t['due_at']) $task_overdue++;
    }
    $pen_count = 0; $pen_sum = 0;
    foreach (($pens['items'] ?? []) as $p) {
        if ((int)($p['employee_id'] ?? 0) !== $uid) continue;
        $ts = (int)($p['final_ok_at'] ?? $p['created_at'] ?? 0);
        if ($ts < $start || $ts > $end) continue;
        $pen_count++;
        $pen_sum += (int)($p['amount'] ?? 0);
    }
    $offish_count = 0; $offish_presence = 0; $offish_absent = 0;
    foreach (($offishes['items'] ?? $offishes) as $o) {
        $ts = (int)($o['start_at'] ?? 0);
        if ($ts < $start || $ts > $end) continue;
        if ((int)($o['photographer_id'] ?? 0) === $uid || (int)($o['blogger_id'] ?? 0) === $uid) {
            $offish_count++;
            if (!empty($o['arrived'][$uid])) $offish_presence++; else $offish_absent++;
        }
    }
    $greet = function_exists('friendly_greeting_v2') ? friendly_greeting_v2() : "سلام";
    $txt = $greet." 👋\n<b>گزارش پرسنل</b>\n";
    $txt .= "کاربر: <b>".$name."</b> (".$uid.")\n";
    $txt .= "بازه: <i>".$period."</i>\n\n";
    $txt .= "🗒️ وظایف: ".$task_done." از ".$task_total." انجام‌شده";
    if ($task_overdue>0) $txt .= " — <b>".$task_overdue." معوق</b>";
    $txt .= "\n🎬 آفیش‌: ".$offish_count." (حضور: ".$offish_presence.", غیبت: ".$offish_absent.")\n";
    $txt .= "🚨 جریمه‌ها: ".$pen_count." مورد — جمع: ".number_format($pen_sum)." تومان\n";
    $txt .= "\n— گزارش خودکار ویژن مدیا —";
    return $txt;
}

function vm_report_send_to(int $chat_id, string $text): void {
    tg_send_html($chat_id, $text);
}

// Send scheduled weekly/monthly reports
function vm_reports_auto(): void {
    // Weekly: every Saturday 20:00 local send prev_week
    $h = (int)date('G'); $i = (int)date('i'); $w = (int)date('w'); // 6=Sat
    if ($w==6 && $h==20 && $i<5) {
        vm_reports_broadcast('prev_week');
    }
    // Monthly: day 1 at 20:00 send prev_month
    if (date('j')==='1' && $h==20 && $i<5) {
        vm_reports_broadcast('prev_month');
    }
}

function vm_reports_broadcast(string $period): void {
    $users = users_all();
    // Collect recipients: all dept managers and all accountants
    $dm_ids = []; $acc_ids = [];
    foreach ($users as $u) {
        $r = $u['role'] ?? '';
        if ($r==='dept_manager') $dm_ids[(int)$u['id']] = true;
        if ($r==='accountant') $acc_ids[(int)$u['id']] = true;
    }
    $emp_ids = [];
    foreach ($users as $u) {
        if (($u['role'] ?? '')==='employee') $emp_ids[] = (int)$u['id'];
    }
    foreach (array_keys($dm_ids) as $rid) {
        foreach ($emp_ids as $eid) {
            $txt = vm_report_employee_text($eid, $period);
            vm_report_send_to($rid, $txt);
        }
    }
    foreach (array_keys($acc_ids) as $rid) {
        foreach ($emp_ids as $eid) {
            $txt = vm_report_employee_text($eid, $period);
            vm_report_send_to($rid, $txt);
        }
    }
}
